/**
* \file: IConfiguration.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: AAUTO
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef AAUTO_ICONFIGURATION_H
#define AAUTO_ICONFIGURATION_H

#include <string>
#include <list>
#include <cstdint>
#include <stdint.h>

namespace adit { namespace aauto
{

class IConfiguration
{
public:
    /**
     * Default constructor.
     */
    IConfiguration();

    /**
     * Default virtual destructor.
     */
    virtual ~IConfiguration() {};

    /**
     * Check if an item exists.
     * Do not use with multi-line items.
     *
     * \param [in] key      key name of the item
     * \param [out] result   value of the item
     *
     * \return bool         true: if available, false: if not available
     */
    virtual bool TryGetItem(const std::string& key, std::string& result) const = 0;

    /**
     * Read configuration item with warning and default.
     * In case there are multiple numbered items with the same key they will be merged to on line.
     * Same as GetMultiItem merged with spaces.
     *
     * \param [in] key      key name of the item
     * \param [in] default  default value
     *
     * \return string       value of the item or defaultValue if not found (with warning)
     */
    std::string GetItem(const std::string& key, const std::string& defaultValue) const;

    /**
     * Read configuration item with more than one line.
     * All items read until next number is not available.
     * example-item-0
     * example-item-1
     *
     * \param [in] key      key name of the item without the number and the "-" before
     *
     * \return list<string> list of the items or empty if nothing found
     */
    virtual std::list<std::string> GetItems(const std::string& key) const = 0;

    /**
     * Read item as an integer with warning and default.
     * Either decimal or hexadecimal with prefix 0x.
     *
     * \param [in] key      key name of the item
     * \param [in] default  default value
     *
     * \return long         number or defaultValue if not found (with warning)
     */
    int64_t GetNumber(const std::string& key, int64_t defaultValue) const;

    /**
     * Extract key and value from node item.
     * If node is empty, empty strings for key and value returned
     *
     * \param [in] inNode      node which contains key and value
     * \param [out] outKey     key as result
     * \param [out] outValue   value as result
     *
     */
    void GetNodeItem(const std::string& inNode, std::string& outKey, std::string& outValue);

    /**
     * Search for first non-space entry (which is properly the key)
     * in input string and return result.
     *
     * \param [in] inString     string which contains the key
     *
     * \return <string> value e.g. key of the item or empty string
     */
    virtual std::string GetKey(const std::string& inString) const = 0;

    /**
     * Search for last non-space entry (which is properly the value)
     * in input string and return result.
     *
     * \param [in] inString     string which contains the value
     *
     * \return <string> value of the item or empty string
     */
    virtual std::string GetValue(const std::string& inString) const = 0;
};

class IDynamicConfiguration : public IConfiguration
{
public:
    /**
     * Default virtual destructor.
     */
    virtual ~IDynamicConfiguration() {};

    /**
     * Write configuration item.
     * The item will be stored only in memory and not reflected to any file.
     *
     * \param [in] key      key name of the item
     * \param [in] value    value of the item
     */
    virtual void SetItem(const std::string& key, const std::string& value) = 0;
};

} } /* namespace adit { namespace aauto */

#endif /* AAUTO_ICONFIGURATION_H */
